document.getElementById('start-button').addEventListener('click', startGame);
document.getElementById('continue-button').addEventListener('click', continueGame);
document.getElementById('save-button').addEventListener('click', saveGame);

// 新規追加: 初期画面のPlayボタン
document.getElementById('initial-play-button').addEventListener('click', showTitleScreen);

let currentBGM = '';
let bgmPosition = 0;
let bgmElement = document.getElementById('bgm');

// タイトル画面のBGMパス（必要に応じて変更）
const titleBGM = './data/bgm/歩む者【大同高校⧸Original BGM】.mp3'; // タイトル画面用のBGMファイルパス

// 変数管理オブジェクト
let gameVariables = {
    scene1VisitedCount: 0, /* スイッチ「あ」の回数のカウントに使う変数。 */
};

// スイッチ管理オブジェクト
let switches = {
/* スイッチ「あ」を定義。最初は実行しないので falseにしておく。 */
    'あ': false, 
};

// 現在のシーンIDをグローバルに保持
let currentSceneId = 0;

// ゲーム開始時にセーブデータの有無を確認し、Continueボタンを有効化
window.onload = function() {
    const savedData = localStorage.getItem('gameSave');
    if (savedData) {
        document.getElementById('continue-button').disabled = false;/* セーブデータが無い場合は、コンティニューボタンを無効化します。 */
    }
};

// 新規追加: 初期画面からタイトル画面へ
function showTitleScreen() {
    document.getElementById('initial-screen').style.display = 'none';
    document.getElementById('title-screen').style.display = 'block';
    // タイトル画面でBGM再生
    changeBGM(titleBGM);
}

/* セーブデータはブラウザのJSONに保存されるため、セーブデータ無しの状態でプレイする場合は、プライベートモード（シークレットモード）でプレイしてください。 */
function startGame() {
    document.getElementById('title-screen').style.display = 'none';
    document.getElementById('game-screen').style.display = 'block';
    showScene(0); /* 開始のシーンid。デバッグ時に、開始したいシーンのidにすることで、その箇所からゲームをプレイできます。 */
}

function continueGame() {
    const savedData = JSON.parse(localStorage.getItem('gameSave'));/* ブラウザのJSONデータをロード */
    if (savedData) {
        document.getElementById('title-screen').style.display = 'none';
        document.getElementById('game-screen').style.display = 'block';
        gameVariables = savedData.gameVariables; // 変数の復元
        switches = savedData.switches; // スイッチの復元
        showScene(savedData.sceneId); // 保存されたシーンに移動
    }
}

function saveGame() {
    const saveData = {
        sceneId: currentSceneId, // グローバル変数からシーンIDを取得
        gameVariables: gameVariables,
        switches: switches
    };
    localStorage.setItem('gameSave', JSON.stringify(saveData));/* ブラウザのJSONにデータを保存 */
    document.getElementById('continue-button').disabled = false;
    alert('ゲームが保存されました！');
}
/* ディレクトリパスはHTMLに代入されます。なので、HTMLの場所を基準にファイルを参照するディレクトリパスを書き込んでください。 */

const scenes = [
    {
        id: 0,
        background: './data/img/background', // 場面の背景画像
        defaultLeftCharacter: './data/img/character/', // 左側に表示する立ち絵
        defaultCenterCharacter: './data/img/character/', // 中央に表示する立ち絵
        defaultRightCharacter: './data/img/character/', // 右側に表示する立ち絵
        dialogue: 'これは最初のシーンです。', // 画面に表示するテキスト
        bgm: './data/bgm/', // BGM
        sfx: './data/se/', // 効果音
        textTickSFX: './data/se/text-se/会話パート0.mp3', // 文字送り用SE
        choices: [
            { text: 'インデックス番号１', nextScene: 1 }, // ボタンを押した際、表示したいシーンのidとボタンに表示するテキスト
            { text: 'インデックス番号0', nextScene: 0 }
        ],
        alternateScenes: {
            'あ': {
                background: './data/img/background',
                defaultLeftCharacter: './data/img/character/', // プロパティ名を統一
                defaultCenterCharacter: './data/img/character/',
                defaultRightCharacter: './data/img/character/',
                dialogue: 'これはスイッチ「あ」がオンの場合のシーンです。',
                bgm: './data/bgm/',
                sfx: './data/se/',
                textTickSFX: './data/se/text-se/会話パート1.mp3', // 文字送り用SE
                choices: [
                    { text: 'インデックス番号３', nextScene: 3 },
                    { text: 'インデックス番号０', nextScene: 0 }
                ]
            }
        }
    },
    {
        id: 1,
        background: './data/img/background',
        defaultLeftCharacter: './data/img/character/',
        defaultCenterCharacter: './data/img/character/',
        defaultRightCharacter: './data/img/character/',
        dialogue: 'インデックス番号１のシーンです。',
        bgm: './data/bgm/',
        sfx: './data/se/',
        textTickSFX: './data/se/text-se/会話パート0.mp3', // 文字送り用SE
        choices: [
            { text: 'インデックス番号１', nextScene: 1 },
            { text: 'インデックス番号２', nextScene: 2 }
        ]
    },
    {
        id: 2,
        background: './data/img/background',
        defaultLeftCharacter: './data/img/character/',
        defaultCenterCharacter: './data/img/character/',
        defaultRightCharacter: './data/img/character/',
        dialogue: 'これはインデックス番号２のシーンです。',
        bgm: './data/bgm/',
        sfx: './data/se/',
        textTickSFX: './data/se/text-se/会話パート0.mp3', // 文字送り用SE
        choices: [
            { text: 'インデックス番号１', nextScene: 1 },
            { text: 'インデックス番号０', nextScene: 0 }
        ]
    },
    {
        id: 3,
        background: './data/img/background',
        defaultLeftCharacter: './data/img/character/',
        defaultCenterCharacter: './data/img/character/',
        defaultRightCharacter: './data/img/character/',
        dialogue: 'ゲームオーバー',
        bgm: './data/bgm/',
        sfx: './data/se/',
        textTickSFX: './data/se/text-se/会話パート1.mp3', // 文字送り用SE
        choices: [
            { text: 'インデックス番号０に戻る', nextScene: 0 },
        ]
    }
];

function showScene(index) {
    const scene = scenes.find(s => s.id === index);
    if (!scene) {
        console.error("Scene not found:", index);
        return;
    }
    currentSceneId = scene.id;

    if (scene.id === 1) {
        gameVariables.scene1VisitedCount++;
    }

    if (gameVariables.scene1VisitedCount >= 5) {
        setSwitch('あ', true);
        gameVariables.scene1VisitedCount = 0;
    }

    if (scene.alternateScenes) {
        for (let key in scene.alternateScenes) {
            if (switches[key]) {
                const alternateScene = scene.alternateScenes[key];
                scene.background = alternateScene.background;
                scene.defaultLeftCharacter = alternateScene.defaultLeftCharacter;
                scene.defaultCenterCharacter = alternateScene.defaultCenterCharacter;
                scene.defaultRightCharacter = alternateScene.defaultRightCharacter;
                scene.dialogue = alternateScene.dialogue;
                scene.bgm = alternateScene.bgm;
                scene.sfx = alternateScene.sfx;
                scene.textTickSFX = alternateScene.textTickSFX; // 文字送りSEも適用
                scene.choices = alternateScene.choices;
                break;
            }
        }
    }

    document.getElementById('background').style.backgroundImage = `url(${scene.background})`;
    document.getElementById('left-character').style.backgroundImage = `url(${scene.defaultLeftCharacter})`;
    document.getElementById('center-character').style.backgroundImage = `url(${scene.defaultCenterCharacter})`;
    document.getElementById('right-character').style.backgroundImage = `url(${scene.defaultRightCharacter})`;
    changeBGM(scene.bgm);
    playSFX(scene.sfx);
    typeDialogue(scene.dialogue, scene.textTickSFX, () => {
        if (scene.choices.length > 0) {
            showChoices(scene.choices);
        }
    });
}

function setSwitch(switchName, state) {
    if (switches.hasOwnProperty(switchName)) {
        switches[switchName] = state;
    }
}

function changeBGM(bgm) {
    if (bgm === currentBGM) {
        bgmElement.currentTime = bgmPosition;
        bgmElement.play().catch(err => console.error("BGM play error:", err));
    } else {
        fadeOut(bgmElement, () => {
            bgmElement.src = bgm;
            bgmElement.currentTime = 0;
            bgmElement.play().catch(err => console.error("BGM play error:", err));
            fadeIn(bgmElement);
            currentBGM = bgm;
            bgmPosition = 0;
        });
    }
}

function fadeIn(audioElement) {
    let volume = 0;
    audioElement.volume = volume;
    const fadeInInterval = setInterval(() => {
        if (volume < 1) {
            volume += 0.1;
            audioElement.volume = volume;
        } else {
            clearInterval(fadeInInterval);
        }
    }, 100);
}

function fadeOut(audioElement, callback) {
    let volume = audioElement.volume;
    const fadeOutInterval = setInterval(() => {
        if (volume > 0) {
            volume -= 0.1;
            audioElement.volume = volume;
        } else {
            clearInterval(fadeOutInterval);
            callback();
        }
    }, 100);
}

function playSFX(sfx) {
    if (sfx) {
        const sfxElement = new Audio(sfx);
        sfxElement.play().catch(err => console.error("SFX play error:", err));
    }
}

bgmElement.addEventListener('timeupdate', () => {
    bgmPosition = bgmElement.currentTime;
});

function typeDialogue(text, textTickSFX, callback) {
    const dialogueText = document.getElementById('dialogue-text');
    dialogueText.textContent = '';
    let i = 0;
    const typingSpeed = 60;
    const interval = setInterval(() => {
        if (i < text.length) {
            dialogueText.textContent += text.charAt(i);
            // 文字送りSEを再生（スペースや空文字でない場合）
            if (textTickSFX && text.charAt(i).trim() !== '') {
                const tickSFX = new Audio(textTickSFX);
                tickSFX.play().catch(err => console.error("Text tick SFX play error:", err));
            }
            i++;
        } else {
            clearInterval(interval);
            setTimeout(callback, 500);
        }
    }, typingSpeed);
}

function showChoices(choices) {
    const choicesContainer = document.getElementById('choices');
    choicesContainer.style.display = 'block';
    choicesContainer.innerHTML = '';
    choices.forEach(choice => {
        const button = document.createElement('button');
        button.className = 'choice-button';
        button.textContent = choice.text;
        button.addEventListener('click', () => {
            choicesContainer.style.display = 'none';
            showScene(choice.nextScene);
        });
        choicesContainer.appendChild(button);
    });
}